package com.greenthrottle.gcs.api;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map.Entry;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;
import android.view.KeyEvent;

public class ControllerEvent implements Parcelable
{

	/**
	 * Indicates type of action.  Primarily used to indicate button down (ACTION_DOWN) and button up (ACTION_UP)
	 *
	 */
	public enum Action
	{
		INVALID,
		UPDATE,
		ACTION_DOWN,
		ACTION_UP,
		STROBE_TO_ON,		// sequence: was On, then toggled Off, then back On
		STROBE_TO_OFF,		// sequence: was Off, then toggled On, then back Off
		RESERVED;
		
		public static Action makeFromInt(int codeValue)
		{
			if(codeValue >= 0 && codeValue < Action.values().length) {
				return Action.values()[codeValue];
			} else {
				return RESERVED; 
			}
		}
	}
	
	/**
	 * Indicates the controller button, stick, or trigger that caused the event
	 *
	 */
	public enum CommonCodes
	{
	
		__FIRST__ (0,"don't use"),
		INVALID (-1,"(invalid)"),
		
		__BUTTONS_BEGIN__ (0,"don't use"),
		A_BUTTON (0,"A Button"),
		B_BUTTON (1,"B Button"),
		C_BUTTON (2,"C Button"),
		D_BUTTON (3,"D Button"),
		X_BUTTON (4, "X Button"),
		Y_BUTTON (5, "Y Button"),
		BUTTON_1 (6, "Button 1"),
		BUTTON_2 (7, "Button 2"),
		BUTTON_3 (8, "Button 3"),
		BUTTON_4 (9, "Button 4"),
		START_BUTTON (15, "Start Button"),
		BACK_BUTTON (16, "Select Button"),
		HOME_BUTTON (17, "Home Button"),
		R1_BUTTON (20, "R1 Button"),
		R2_BUTTON (21, "R2 Button"),
		R3_BUTTON (22, "R3 Button"),
		
		L1_BUTTON (23, "L1 Button"),
		L2_BUTTON (24, "L2 Button"),
		L3_BUTTON (25, "L3 Button"),
		__BUTTONS_END__ (25, "don't use"),
		
		__DPAD_BEGIN__ (30, "don't use"),
		DPAD_UP (30, "DPAD Up"),
		DPAD_RIGHT (31, "DPAD Right"),
		DPAD_DOWN (32, "DPAD Down"),
		DPAD_LEFT (33, "DPAD Left"),
		__DPAD_END__ (33, "don't use"),
		
		__ANALOG_DPAD_BEGIN__ (40, "don't use"),
		LEFT_ANALOG_AS_DPAD_UP (40, "LEFT ANALOG-as-DPAD Up"),
		LEFT_ANALOG_AS_DPAD_RIGHT (41, "LEFT ANALOG-as-DPAD Right"),
		LEFT_ANALOG_AS_DPAD_DOWN (42, "LEFT ANALOG-as-DPAD Down"),
		LEFT_ANALOG_AS_DPAD_LEFT (43, "LEFT ANALOG-as-DPAD Left"),
		
		RIGHT_ANALOG_AS_DPAD_UP (50, "RIGHT ANALOG-as-DPAD Up"),
		RIGHT_ANALOG_AS_DPAD_RIGHT (51, "RIGHT ANALOG-as-DPAD Right"),
		RIGHT_ANALOG_AS_DPAD_DOWN (52, "RIGHT ANALOG-as-DPAD Down"),
		RIGHT_ANALOG_AS_DPAD_LEFT (53, "RIGHT ANALOG-as-DPAD Left"),
		__ANALOG_DPAD_END__ (53, "don't use"),
		
		__ANALOG_BEGIN__ (60, "don't use--convenience"),
		LEFT_ANALOG (60, "LEFT ANALOG STICK"),
		RIGHT_ANALOG (61, "RIGHT_ANALOG_STICK"),
		L2_ANALOG (62, "L2 ANALOG"),
		R2_ANALOG (63, "R2 ANALOG"),
		DIAL_ANALOG (64, "DIAL ANALOG"),
		__ANALOG_END__ (64, "don't use--convenience"),
		__LAST__ (64,"don't use"),
		
		
		RESERVED (1000, "message is too new for this API version");
		
		CommonCodes(int codeValue,String displayName)
		{
			val = codeValue;
			name = displayName;
		}
		
		/**
		 * Used internally as part of IPC to regenerate event object
		 * @param codeValue
		 * @return
		 */
		public static CommonCodes makeFromInt(int codeValue)
		{
			if(codeValue >= 0 && codeValue < CommonCodes.values().length) {
				return CommonCodes.values()[codeValue];
			} else {
				return RESERVED; 
			}
		}

		/**
		 * Used internally as part of IPC to regenerate event object
		 * @param codeValue
		 * @return
		 */
		public static CommonCodes[] makeFromIntAllAlternatives(int codeValue)
		{
			//this variant matches ALL of the codes for the value, in case multiple, aliased...
			if ((codeValue < __FIRST__.v()) || (codeValue > __LAST__.v()))
			{ return null; }
			
			ArrayList<CommonCodes> _list = new ArrayList<CommonCodes>();
			
			for (CommonCodes c : CommonCodes.values())
			{
				if (codeValue == c.v())
				{ _list.add(c); }
			}
			
			if (_list.size() > 0)
			{
				CommonCodes[] rv = new CommonCodes[_list.size()];
				rv = _list.toArray(rv);
				return rv;
			}
			return null;
		}
		
		public int v()
		{ return val; }
		public String n()
		{ return name; }
		
		public boolean isEqual(Object o)
		{
			if (o == null)
			{ return false; }
			if (o instanceof CommonCodes == false)
			{ return false; }
			return (v() == ((CommonCodes)o).v());
		}
		
		int val;
		String name;
	}
	
	/**
	 * Indicates type of analog event.  X_DATA and Y_DATA are a value from -127...127 
	 * (with axis polarity matching the Cartesian plane) 
	 *
	 */
	public enum AnalogDataType
	{
		INVALID,
		X_MIN,
		X_MAX,
		Y_MIN,
		Y_MAX,
		Z_MIN,
		Z_MAX,
		TWIST_MIN,
		TWIST_MAX,
		
		X_DATA,
		Y_DATA,
		Z_DATA,
		TWIST_DATA,
		RESERVED;
		
		public static AnalogDataType makeFromInt(int codeValue)
		{
			if(codeValue >= 0 && codeValue < AnalogDataType.values().length) {
				return AnalogDataType.values()[codeValue];
			} else {
				return RESERVED; 
			}
		}
	}
	
	/**
	 * Used internally as part of IPC
	 */
	public static final Parcelable.Creator<ControllerEvent> CREATOR
	= new Parcelable.Creator<ControllerEvent>() {
		public ControllerEvent createFromParcel(Parcel in) {
			return new ControllerEvent(in);
		}

		public ControllerEvent[] newArray(int size) {
			return new ControllerEvent[size];
		}
	};
	
	private ControllerEvent(Parcel in) {
		m_controllerId = in.readString();
		m_action = Action.makeFromInt(in.readInt());
		m_cCode = CommonCodes.makeFromInt(in.readInt());
		//in.readMap(m_optional_analogData, HashMap.class.getClassLoader());
		
		m_optional_analogData = new HashMap<AnalogDataType,Double>();
		int analogParams = in.readInt(); 
		if (analogParams >= 1) {
			m_optional_analogData.put(AnalogDataType.X_DATA, in.readDouble());
		}
		if (analogParams == 2) {
			m_optional_analogData.put(AnalogDataType.Y_DATA, in.readDouble());
		}
	}
	
	public ControllerEvent(String controller_id,final Action action,final CommonCodes commonCode)
	{
		m_controllerId = controller_id;
		m_action = action;
		m_cCode = commonCode;
	}
	
	@SuppressWarnings("unchecked")
	public ControllerEvent(String controller_id,CommonCodes analogCode,HashMap<AnalogDataType,Double> analogValues,boolean fillInRelevantMinMaxDefaults)
	{
		m_controllerId = controller_id;
		m_action = Action.UPDATE;
		m_cCode = analogCode;	//and don't pass in anything else...
		m_optional_analogData = (analogValues != null ? (HashMap<AnalogDataType,Double>)analogValues.clone() 
								: new HashMap<AnalogDataType,Double>());
		
		if (fillInRelevantMinMaxDefaults)
		{
			if (m_optional_analogData.containsKey(AnalogDataType.X_DATA) 
					&& ((m_optional_analogData.containsKey(AnalogDataType.X_MAX) == false)
							|| (m_optional_analogData.containsKey(AnalogDataType.X_MIN) == false)
							)
				)
			{
				//add x axis max and min
				m_optional_analogData.put(AnalogDataType.X_MIN,new Double(Default_X_Axis_Min));
				m_optional_analogData.put(AnalogDataType.X_MAX,new Double(Default_X_Axis_Max));
				
			}
			if (m_optional_analogData.containsKey(AnalogDataType.Y_DATA) 
					&& ((m_optional_analogData.containsKey(AnalogDataType.Y_MAX) == false)
							|| (m_optional_analogData.containsKey(AnalogDataType.Y_MIN) == false)
							)
				)
			{
				//add y axis max and min
				m_optional_analogData.put(AnalogDataType.Y_MIN,new Double(Default_Y_Axis_Min));
				m_optional_analogData.put(AnalogDataType.Y_MAX,new Double(Default_Y_Axis_Max));
			}
			if (m_optional_analogData.containsKey(AnalogDataType.Z_DATA) 
					&& ((m_optional_analogData.containsKey(AnalogDataType.Z_MAX) == false)
							|| (m_optional_analogData.containsKey(AnalogDataType.Z_MIN) == false)
							)
				)
			{
				//add z axis max and min
				m_optional_analogData.put(AnalogDataType.Z_MIN,new Double(Default_Z_Axis_Min));
				m_optional_analogData.put(AnalogDataType.Z_MAX,new Double(Default_Z_Axis_Max));
			}
			if (m_optional_analogData.containsKey(AnalogDataType.TWIST_DATA) 
					&& ((m_optional_analogData.containsKey(AnalogDataType.TWIST_MAX) == false)
							|| (m_optional_analogData.containsKey(AnalogDataType.TWIST_MIN) == false)
							)
				)
			{
				//add twist axis max and min
				m_optional_analogData.put(AnalogDataType.TWIST_MIN,new Double(Default_Twist_Axis_Min));
				m_optional_analogData.put(AnalogDataType.TWIST_MAX,new Double(Default_Twist_Axis_Max));
			}
		}
	}
	
	@Override
	public  String toString()
	{
		StringBuilder sb = new StringBuilder();
		sb.append("{ ControllerId: "+m_controllerId);
		if (m_cCode != null)
		{
			sb.append(", Action: "+m_action.toString()+" . Which: " + m_cCode.n());
		}
		
		sb.append(" , ");
		sb.append(analogValuesToString());
		sb.append(" }");
		return sb.toString();
	}
	
	/**
	 * Provide string representation of analog values for debugging
	 * @return
	 */
	public  String analogValuesToString()
	{
		if (m_optional_analogData == null)
		{
			return "";
		}
		StringBuilder sb = new StringBuilder("analog_data = { ");
		for (HashMap.Entry<AnalogDataType,Double> entry : m_optional_analogData.entrySet()) {
		    sb.append("(");
			sb.append(entry.getKey().toString());
		    sb.append(" , ");
		    sb.append(entry.getValue());
		    sb.append(") ");
		}
		sb.append(" }");
		return sb.toString();
	}
	
	public  String toCodeString()
	{
		if (m_cCode != null)
		{
			return "{ " + m_cCode.n() + " }";
		}
		return "{}";
	}
	
	
	public  boolean isValid()
	{
		return (m_action.equals(Action.INVALID) == false);
	}
	
	/**
	 * Returns the BT address of the controller that generated this event
	 * @return
	 */
	public  String id()
	{
		return m_controllerId;
	}
	
	/**
	 * Returns Action that triggered this event, such as a button up or button down
	 * @return Action
	 */
	public  Action action()
	{
		return m_action;
	}
	
	/**
	 * Returns analog data associated with this event, such as the x & y axis positions of one of the
	 * analog sticks
	 * @return
	 */
	public HashMap<AnalogDataType,Double>  analogData() {
		return m_optional_analogData;
	}
	
	/**
	 * Used internally
	 * @return
	 */
	public int code()
	{
		if (m_cCode != null)
		{ return m_cCode.v(); }
		
		return CommonCodes.INVALID.v();
	}
	
	/**
	 * Returns the CommonCode corresponding with the controller button, stick, or trigger
	 * that generated this event
	 * @return
	 */
	public CommonCodes cCode()
	{
		return m_cCode;
	}
	
	/**
	 * Used internally
	 * @param v
	 */
	public void changeCommonCode(CommonCodes v)
	{
		m_cCode = v;
	}

	/**
	 * Unused
	 * @return
	 */
	public  KeyEvent keyEvent()
	{
		return m_optional_assocKeyEvent;
	}
	
	protected String m_controllerId;
	protected Action m_action;
	protected CommonCodes m_cCode;	//if commoncode used
	protected KeyEvent m_optional_assocKeyEvent;
	protected HashMap<AnalogDataType,Double> m_optional_analogData;
	
	protected static Double Default_X_Axis_Min = Double.valueOf(-127.0);
	protected static Double Default_Y_Axis_Min = Double.valueOf(-127.0);
	protected static Double Default_Z_Axis_Min = Double.valueOf(-127.0);
	protected static Double Default_Twist_Axis_Min = Double.valueOf(-127.0);
	
	protected static Double Default_X_Axis_Max = Double.valueOf(127.0);
	protected static Double Default_Y_Axis_Max = Double.valueOf(127.0);
	protected static Double Default_Z_Axis_Max = Double.valueOf(127.0);
	protected static Double Default_Twist_Axis_Max = Double.valueOf(127.0);
	
	/**
	 * Used internally by IPC
	 * @return
	 */
	public int describeContents() {
		// TODO Auto-generated method stub
		return 0;
	}

	/**
	 * Used internally by IPC
	 * @param out
	 * @param arg1
	 */
	public void writeToParcel(Parcel out, int arg1) {
		// TODO Auto-generated method stub
		out.writeString(m_controllerId);
		out.writeInt(m_action.ordinal());
		out.writeInt(m_cCode.ordinal());
		
		int analogLen = 0;
		if (m_optional_analogData != null) {
			analogLen = m_optional_analogData.size();
		}
		if (analogLen > 0) {
			out.writeInt(analogLen);
			out.writeDouble(m_optional_analogData.get(AnalogDataType.X_DATA));
			if (analogLen == 2) {
				out.writeDouble(m_optional_analogData.get(AnalogDataType.Y_DATA));
			}
		} else {
			out.writeInt(0);
		}
		
	}

	public void setId(String bt_address) {
		m_controllerId = bt_address;
	}
}
