package com.greenthrottle.unifier;

import com.greenthrottle.gcs.api.ControllerEvent;
import com.greenthrottle.gcs.api.ControllerInfo;
import com.greenthrottle.gcs.api.IGTController;
import com.greenthrottle.gcs.api.IGTControllerCallback;
import com.greenthrottle.gcs.api.ServiceStatusEvent;
import com.greenthrottle.gcs.api.ControllerEvent.AnalogDataType;

import java.util.HashMap;
import java.util.Map;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Log;
import android.view.View;

public abstract class GreenThrottleService 
{
	boolean mIsBound = false;
	boolean m_serviceIsConnected = false;

	/** The primary interface we will be calling on the service. */
	IGTController mService = null;

	public Map<String,ControllerPlayer> m_boundPlayerControllers = new HashMap<String,ControllerPlayer>();

	private HashMap<ControllerEvent.CommonCodes,ControllerEvent.CommonCodes> simpleCodeRemap = null;

	public GreenThrottleService()
	{
		simpleCodeRemap = 
				new HashMap<ControllerEvent.CommonCodes,ControllerEvent.CommonCodes>();
	}
	
	///////////////////////////////////////////////////////
	public void BindService(Activity activity) 
	{
		if (!mIsBound)
		{
			mIsBound = activity.bindService(new Intent(IGTController.class.getName()),
					mConnection, Context.BIND_AUTO_CREATE);
		}
	}

	public void UnbindService(Activity activity) 
	{
		if (mIsBound) {
			// If we have received the service, and hence registered with
			// it, then now is the time to unregister.
			if (mService != null) 
			{
				try 
				{
					mService.unregisterCallback(mCallback);
				} catch (RemoteException e) {
					// There is nothing special we need to do if the service
					// has crashed.
				}
			}

			// Detach our existing connection.
			activity.unbindService(mConnection);
			mIsBound = false;
		}
	}

	public void hideSystemUI(Activity activity)
	{
		// Ask the System Bar to hide
		try 
		{
			View mainLayout = activity.getWindow().getDecorView().getRootView();
			mainLayout.setSystemUiVisibility(View.SYSTEM_UI_FLAG_HIDE_NAVIGATION|View.SYSTEM_UI_FLAG_LOW_PROFILE);
		} 
		catch (NoSuchMethodError ex) 
		{
			Log.i("GreenThrottleService", "We are not ICS");
		}

	}
	
	public void queryCurrentBindings()
	{
		if (mIsBound) 
		{
			if (mService != null) 
			{
				try 
				{
					//make up a random number for the TID
					mService.queryCurrentBindings();
				} catch (RemoteException e) {
					// There is nothing special we need to do if the service
					// has crashed.
				}
			}
		}
	}

	public void addLeftRemap()
	{
		if( !simpleCodeRemap.containsKey(ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_DOWN))
		{
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_DOWN,
					ControllerEvent.CommonCodes.DPAD_DOWN);
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_LEFT,
					ControllerEvent.CommonCodes.DPAD_LEFT);
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_UP,
					ControllerEvent.CommonCodes.DPAD_UP);
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_RIGHT,
					ControllerEvent.CommonCodes.DPAD_RIGHT);
		}
	}
	
	public void removeLeftRemap()
	{
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_DOWN);
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_LEFT);
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_UP);
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.LEFT_ANALOG_AS_DPAD_RIGHT);
	}
	
	public void addRightRemap()
	{
		if( !simpleCodeRemap.containsKey(ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_DOWN))
		{
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_DOWN,
					ControllerEvent.CommonCodes.A_BUTTON);
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_LEFT,
					ControllerEvent.CommonCodes.X_BUTTON);
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_UP,
					ControllerEvent.CommonCodes.Y_BUTTON);
			simpleCodeRemap.put(
					ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_RIGHT,
					ControllerEvent.CommonCodes.B_BUTTON);
		}
	}
	
	public void removeRightRemap()
	{
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_DOWN);
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_LEFT);
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_UP);
		simpleCodeRemap.remove(ControllerEvent.CommonCodes.RIGHT_ANALOG_AS_DPAD_RIGHT);
	}

	//get info about button presses
	protected abstract void ButtonAction(ControllerPlayer player, ControllerEvent.CommonCodes code, boolean pressed);
	//get info about analog activity
	protected abstract void AnalogEvent(ControllerPlayer player, ControllerEvent.CommonCodes code, float x, float y);
	//get info about controller connection activity
	protected abstract void ControllerAction(ControllerPlayer player, boolean connected);
	//whether or not the service is connected
	protected abstract void ServiceStatus(boolean isConnected);
	
	private ServiceConnection mConnection = new ServiceConnection() 
	{
		public void onServiceConnected(ComponentName className,
				IBinder service) 
		{
			// This is called when the connection with the service has been
			// established, giving us the service object we can use to
			// interact with the service.  We are communicating with our
			// service through an IDL interface, so get a client-side
			// representation of that from the raw service object.
			mService = IGTController.Stub.asInterface(service);

			// We want to monitor the service for as long as we are
			// connected to it.
			try 
			{
				mService.registerCallback(mCallback);
			} 
			catch (RemoteException e) 
			{
				// In this case the service has crashed before we could even
				// do anything with it; we can count on soon being
				// disconnected (and then reconnected if it can be restarted)
				// so there is no need to do anything here.
				e.printStackTrace();
			}
			m_serviceIsConnected = true;
			ServiceStatus(true);
			
			doStartAutoconnect();
			queryCurrentBindings();
			
			// As part of the sample, tell the user what happened.
			//Toast.makeText(this, "GCS connected",
			//		Toast.LENGTH_SHORT).show();
		}

		public void onServiceDisconnected(ComponentName className) 
		{
			// This is called when the connection with the service has been
			// unexpectedly disconnected -- that is, its process crashed.
			mService = null;
			m_serviceIsConnected = false;
			ServiceStatus(false);
			
			// As part of the sample, tell the user what happened.
			//Toast.makeText(GreenThrottleService.this, "GCS DIS-connected",
			//		Toast.LENGTH_SHORT).show();
		}
	};

	private IGTControllerCallback mCallback = new IGTControllerCallback.Stub() 
	{
		/**
		 * This is called by the remote service regularly to tell us about
		 * new values.  Note that IPC calls are dispatched through a thread
		 * pool running in each process, so the code executing here will
		 * NOT be running in our main thread like most other things -- so,
		 * to update the UI, we need to use a Handler to hop over there.
		 */

		public void controllerEvent(ControllerEvent e) throws RemoteException 
		{
			handleControllerEvent(e);
		}

		public void serviceStatusUpdateEvent(ServiceStatusEvent e) throws RemoteException 
		{
			handleServiceStatusEvent(e);
		}

		/**
		 * response:   "OK"  ,   "REJECTED"
		 * extended_response (if not null or ""):  the TID this client specified when it made the request
		 */
		@Override
		public void commandAck(String response,String extended_response) throws RemoteException 
		{
			if (response == null)
			{ return; } //guard
			//Log.i("GreenThrottleService", "response = "+response+(extended_response == null ? "" : (" , extended response = "+extended_response)));
		}

		@Override
		public void controllerInfo(ControllerInfo[] ia)
				throws RemoteException 
		{
			handleControllerInfoMessage(ia);
		}
	};
	
	private void doStartAutoconnect() 
	{
		if (mIsBound) 
		{
			if (mService != null) 
			{
				try 
				{
					mService.startAutoconnect();
				} catch (RemoteException e) 
				{
					// There is nothing special we need to do if the service
					// has crashed.
				}
			}
		}
	}
	
	private void handleServiceStatusEvent(ServiceStatusEvent e)
	{
		if (e == null)
		{ return; }
		switch (e.code())
		{
		case CONTROLLER_DISCONNECTED:
			controllerDisconnected(e.encodedData());
			break;
		case CONTROLLER_CONNECTED:
		case QUERY_BOUND_CONTROLLERS_VALUE:
			//EXAMPLE-CODE: just print it ...but useful code will want to process the mapping and utilize it somehow
			Log.i("GreenThrottleService","mapping received: "+( e.encodedData() == null ? "(null)" : e.encodedData()));

			controllerConnected(e.encodedData());
			break;
		default:
			//Log.w("GreenThrottleService","unhandled code: ["+e.toString()+"]");
			break;
		}
	}

	private void controllerConnected(String controller_bt_address)
	{
		if(controller_bt_address.contains("="))
		{
			// controller is bound
			String[] list = controller_bt_address.split("=");
			controller_bt_address = list[0];
			String player = list[1];

			list = player.split("_");
			int playerNumberAssigned = Integer.parseInt(list[1]);
			ControllerPlayer ctrlp = new ControllerPlayer(playerNumberAssigned);
			m_boundPlayerControllers.put(controller_bt_address,ctrlp.bindControllerToMe(controller_bt_address));
			ControllerAction(ctrlp, true);
		}
		else
		{
			// controller is not bound
		}
	}
	
	private void controllerDisconnected(String controller_bt_address)
	{
		//remove from unbound, and remove from player entry
		//TODO: handle disconnect better in case it reconnects shortly after
		ControllerPlayer ctrlp = m_boundPlayerControllers.remove(controller_bt_address);
		if (ctrlp != null)
		{
			ControllerAction(ctrlp, false);
		}
	}
	
	private void handleControllerInfoMessage(ControllerInfo[] ia)
	{

		
		Log.i("GreenThrottleService","message received from service...controller info data follows:");
		if (ia == null)
		{
			Log.w("GreenThrottleService","ControllerInfo array is null!");
			return;
		}
		if (ia.length == 0)
		{
			Log.w("GreenThrottleService","ControllerInfo array length is 0");
			return;
		}
		for (ControllerInfo ctrlInfo : ia)
		{
			Log.i("GreenThrottleService","controller bt address = "+ctrlInfo.btAddress()
												+ " controller name = "+(ctrlInfo.name() == null ? "(null)" : ctrlInfo.name())
												+ " controller cname = "+(ctrlInfo.sysCName() == null ? "(null)" : ctrlInfo.sysCName())
												+ " status = "+ctrlInfo.state().toString()
					);
		}
		Log.i("GreenThrottleService","(no more controller info objects in message)");
	}
	
	//_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-

	//_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-

	
	private void handleControllerEvent(ControllerEvent e)
	{
		if (e == null)
		{ return; } //it's public so gonna take some caution...

		e = remap(e);
		int code = e.cCode().v();
		if (((code >= ControllerEvent.CommonCodes.__DPAD_BEGIN__.v())
				&& (code <= ControllerEvent.CommonCodes.__DPAD_END__.v()))
				|| ((code >= ControllerEvent.CommonCodes.__BUTTONS_BEGIN__.v())
				&& (code <= ControllerEvent.CommonCodes.__BUTTONS_END__.v()))
				|| ((code >= ControllerEvent.CommonCodes.__ANALOG_DPAD_BEGIN__.v() 
				&&  code <= ControllerEvent.CommonCodes.__ANALOG_DPAD_END__.v())))
		{
			//button event
			controllerButtonEvent(e);
		}
		else if (code >= ControllerEvent.CommonCodes.__ANALOG_BEGIN__.v() 
				&&  code <= ControllerEvent.CommonCodes.__ANALOG_END__.v())
		{
			controllerAnalogEvent(e);
		} 
	}

	private ControllerEvent remap(ControllerEvent e)
	{
		if (e.cCode() == null)
		{ 
			return e; 
		}
		
		ControllerEvent.CommonCodes em = this.simpleCodeRemap.get(e.cCode());
		if (em != null)
		{
			e.changeCommonCode(em);
		}
		return e;
	}
	
	private void controllerAnalogEvent(ControllerEvent e) 
	{
		//find the player that the event is meant for
		ControllerPlayer player = m_boundPlayerControllers.get(e.id());
		if (player == null)
		{ 
			Log.i("GreenThrottleService", "no player for this event"); 
			return; 
		} //no player corresponds to this controller
		
		float x = 0;
		float y = 0;
		HashMap<AnalogDataType,Double> aValues = e.analogData();
		switch (e.cCode()) {
		case LEFT_ANALOG:
		case RIGHT_ANALOG:
			x = aValues.get(ControllerEvent.AnalogDataType.X_DATA).floatValue();
			y = aValues.get(ControllerEvent.AnalogDataType.Y_DATA).floatValue();
			break;
		case L2_ANALOG:
		case R2_ANALOG:
			x = aValues.get(ControllerEvent.AnalogDataType.X_DATA).floatValue();
			break;
		}
		
		AnalogEvent(player, e.cCode(), x, y);
	}

	private void controllerButtonEvent(ControllerEvent e)
	{
		//find the player that the event is meant for
		ControllerPlayer player = m_boundPlayerControllers.get(e.id());
		if (player == null)
		{ 
			Log.i("GreenThrottleService", "no player for this event"); 
			return; 
		} //no player corresponds to this controller

		if (e.action().equals(ControllerEvent.Action.ACTION_DOWN) || e.action().equals(ControllerEvent.Action.ACTION_UP))
		{
			boolean isDown = e.action().equals(ControllerEvent.Action.ACTION_DOWN);
			ButtonAction(player, e.cCode(), isDown);
		}
	}


} // %%%%%% END CLASS %%%%%%%%
